// Background service worker - Upwork Pro v6.3
// Processes AI in background and saves results to storage

const API_URL = 'https://upwork-pro-api.muh-has311.workers.dev';

chrome.runtime.onInstalled.addListener(() => {
  console.log('Upwork Pro v6.3 installed!');
  // Clear any old pending tasks
  chrome.storage.local.remove(['pendingTask', 'taskResult']);
});

// Listen for messages from popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  
  if (request.action === 'startAITask') {
    // Start AI task in background
    const taskData = request.data;
    
    // Save task as pending
    chrome.storage.local.set({ 
      pendingTask: {
        id: taskData.taskId,
        type: taskData.type,
        startTime: Date.now(),
        status: 'processing'
      }
    });
    
    // Process in background
    processAITask(taskData);
    
    sendResponse({ started: true, taskId: taskData.taskId });
    return true;
  }
  
  if (request.action === 'getTaskStatus') {
    chrome.storage.local.get(['pendingTask', 'taskResult'], (data) => {
      sendResponse({
        pending: data.pendingTask || null,
        result: data.taskResult || null
      });
    });
    return true;
  }
  
  if (request.action === 'clearTaskResult') {
    chrome.storage.local.remove(['pendingTask', 'taskResult']);
    sendResponse({ cleared: true });
    return true;
  }
});

async function processAITask(taskData) {
  const { taskId, type, prompt, maxTokens, token } = taskData;
  
  console.log('Background: Processing task', taskId, type);
  
  try {
    const response = await fetch(`${API_URL}/api/ai/generate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      },
      body: JSON.stringify({
        prompt: prompt,
        max_tokens: maxTokens || 4096
      })
    });
    
    const result = await response.json();
    
    console.log('Background: Task complete', taskId);
    
    // Save result to storage
    await chrome.storage.local.set({
      pendingTask: null,
      taskResult: {
        taskId: taskId,
        type: type,
        success: result.success,
        content: result.content,
        credits: result.credits,
        error: result.error,
        completedAt: Date.now()
      }
    });
    
    // Update user credits in storage
    if (result.success && result.credits !== undefined) {
      chrome.storage.local.get(['user'], (data) => {
        if (data.user) {
          data.user.credits = result.credits;
          chrome.storage.local.set({ user: data.user });
        }
      });
    }
    
    // Try to notify popup (might fail if closed)
    try {
      chrome.runtime.sendMessage({ 
        action: 'taskComplete', 
        taskId: taskId,
        result: result 
      });
    } catch (e) {
      console.log('Popup closed, result saved to storage');
    }
    
  } catch (error) {
    console.error('Background: Task error', error);
    
    // Save error to storage
    await chrome.storage.local.set({
      pendingTask: null,
      taskResult: {
        taskId: taskId,
        type: type,
        success: false,
        error: error.message,
        completedAt: Date.now()
      }
    });
  }
}
