// API Service for Upwork Pro v6.1 - FIXED ROUTES
var APIService = (function() {
  var user = null;
  var token = null;
  var installId = null;
  var fingerprint = null;

  async function init() {
    return new Promise(function(resolve) {
      chrome.storage.local.get(['user', 'token', 'installId', 'fingerprint'], function(r) {
        user = r.user || null;
        token = r.token || null;
        installId = r.installId || generateId();
        fingerprint = r.fingerprint || generateFingerprint();
        chrome.storage.local.set({ installId: installId, fingerprint: fingerprint });
        resolve(!!token);
      });
    });
  }

  function generateId() {
    return 'UP' + Date.now().toString(36) + Math.random().toString(36).substr(2, 9);
  }

  function generateFingerprint() {
    var data = [
      navigator.userAgent,
      navigator.language,
      screen.width + 'x' + screen.height,
      new Date().getTimezoneOffset()
    ].join('|');
    var hash = 0;
    for (var i = 0; i < data.length; i++) {
      hash = ((hash << 5) - hash) + data.charCodeAt(i);
      hash = hash & hash;
    }
    return 'FP' + Math.abs(hash).toString(16);
  }

  async function request(endpoint, data) {
    var fullUrl = CONFIG.API_URL + endpoint;
    console.log('=== API REQUEST ===');
    console.log('URL:', fullUrl);
    console.log('Data:', data);
    
    try {
      var response = await fetch(fullUrl, {
        method: 'POST',
        headers: { 
          'Content-Type': 'application/json',
          'Authorization': token ? 'Bearer ' + token : ''
        },
        body: JSON.stringify(Object.assign({}, data, {
          installation_id: installId,
          browser_fingerprint: fingerprint
        }))
      });
      
      console.log('Response Status:', response.status);
      var text = await response.text();
      console.log('Response Text:', text);
      
      try {
        var result = JSON.parse(text);
        console.log('Parsed Result:', result);
        return result;
      } catch (e) {
        console.error('API returned invalid JSON:', text.substring(0, 200));
        return { success: false, error: 'Server error: ' + text.substring(0, 100) };
      }
    } catch (e) {
      console.error('API Network Error:', e);
      return { success: false, error: 'Network error: ' + e.message };
    }
  }

  async function login(email, password) {
    var r = await request('/api/login', { email: email, password: password });
    if (r.success && r.token) {
      token = r.token;
      user = r.user;
      chrome.storage.local.set({ token: token, user: user });
    }
    return r;
  }

  async function signup(name, email, password, phone) {
    var r = await request('/api/signup', { name: name, email: email, password: password, phone: phone });
    if (r.success && r.token) {
      token = r.token;
      user = r.user;
      chrome.storage.local.set({ token: token, user: user });
    }
    return r;
  }

  async function verifyOtp(email, code) {
    var r = await request('/api/verify-otp', { email: email, code: code });
    if (r.success && r.token) {
      token = r.token;
      user = r.user;
      chrome.storage.local.set({ token: token, user: user });
    }
    return r;
  }

  async function resendOtp(email) {
    return await request('/api/resend-otp', { email: email });
  }

  async function forgotPassword(email) {
    return await request('/api/forgot-password', { email: email });
  }

  async function resetPassword(email, code, newPassword) {
    return await request('/api/reset-password', { email: email, code: code, password: newPassword });
  }

  async function verify() {
    if (!token) return { success: false };
    var r = await request('/api/verify', {});
    if (r.success && r.user) {
      user = r.user;
      chrome.storage.local.set({ user: user });
    }
    return r;
  }

  async function logout() {
    token = null;
    user = null;
    chrome.storage.local.remove(['token', 'user']);
    return { success: true };
  }

  // Generate unique task ID
  function generateTaskId() {
    return 'task_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
  }

  // Check for pending/completed task from background
  async function checkPendingTask() {
    return new Promise((resolve) => {
      chrome.runtime.sendMessage({ action: 'getTaskStatus' }, (response) => {
        resolve(response || { pending: null, result: null });
      });
    });
  }

  // Clear task result
  async function clearTaskResult() {
    return new Promise((resolve) => {
      chrome.runtime.sendMessage({ action: 'clearTaskResult' }, resolve);
    });
  }

  // AI Generation via Background Script
  async function generateAI(prompt, maxTokens, taskType) {
    if (!token) return { success: false, error: 'Not logged in' };
    
    // First check if there's already a completed result waiting
    const status = await checkPendingTask();
    
    if (status.result && status.result.success) {
      // There's a completed result - return it and clear
      const result = status.result;
      await clearTaskResult();
      
      // Update credits
      if (result.credits !== undefined) {
        user.credits = result.credits;
        chrome.storage.local.set({ user: user });
      }
      
      return {
        success: true,
        content: result.content,
        credits: result.credits
      };
    }
    
    if (status.result && !status.result.success) {
      // There's an error result - return it and clear
      const result = status.result;
      await clearTaskResult();
      return { success: false, error: result.error };
    }
    
    // Check if task is still processing
    if (status.pending && status.pending.status === 'processing') {
      // Task is running - wait for it
      return await waitForTaskCompletion(status.pending.id);
    }
    
    // Start new task
    const taskId = generateTaskId();
    
    return new Promise((resolve) => {
      // Start background task
      chrome.runtime.sendMessage({
        action: 'startAITask',
        data: {
          taskId: taskId,
          type: taskType || 'generate',
          prompt: prompt,
          maxTokens: maxTokens || 4096,
          token: token
        }
      });
      
      // Listen for completion
      const messageListener = (message) => {
        if (message.action === 'taskComplete' && message.taskId === taskId) {
          chrome.runtime.onMessage.removeListener(messageListener);
          
          if (message.result.credits !== undefined) {
            user.credits = message.result.credits;
            chrome.storage.local.set({ user: user });
          }
          
          resolve(message.result);
        }
      };
      
      chrome.runtime.onMessage.addListener(messageListener);
      
      // Also poll storage in case popup was closed/reopened
      const pollForResult = setInterval(async () => {
        const currentStatus = await checkPendingTask();
        
        if (currentStatus.result) {
          clearInterval(pollForResult);
          chrome.runtime.onMessage.removeListener(messageListener);
          
          const result = currentStatus.result;
          await clearTaskResult();
          
          if (result.credits !== undefined) {
            user.credits = result.credits;
            chrome.storage.local.set({ user: user });
          }
          
          resolve({
            success: result.success,
            content: result.content,
            credits: result.credits,
            error: result.error
          });
        }
      }, 1000);
      
      // Timeout after 3 minutes
      setTimeout(() => {
        clearInterval(pollForResult);
        chrome.runtime.onMessage.removeListener(messageListener);
        resolve({ success: false, error: 'Request timeout' });
      }, 180000);
    });
  }

  // Wait for existing task to complete
  async function waitForTaskCompletion(taskId) {
    return new Promise((resolve) => {
      const pollForResult = setInterval(async () => {
        const status = await checkPendingTask();
        
        if (status.result) {
          clearInterval(pollForResult);
          
          const result = status.result;
          await clearTaskResult();
          
          if (result.credits !== undefined) {
            user.credits = result.credits;
            chrome.storage.local.set({ user: user });
          }
          
          resolve({
            success: result.success,
            content: result.content,
            credits: result.credits,
            error: result.error
          });
        }
        
        if (!status.pending) {
          clearInterval(pollForResult);
          resolve({ success: false, error: 'Task not found' });
        }
      }, 1000);
      
      // Timeout after 3 minutes
      setTimeout(() => {
        clearInterval(pollForResult);
        resolve({ success: false, error: 'Request timeout' });
      }, 180000);
    });
  }

  function getUser() { return user; }
  function getInstallId() { return installId; }
  function getFingerprint() { return fingerprint; }
  function hasCredits() { return user && user.credits > 0; }
  function isPremium() { return user && user.total_purchased > 0; }

  return {
    init: init,
    login: login,
    signup: signup,
    verifyOtp: verifyOtp,
    resendOtp: resendOtp,
    forgotPassword: forgotPassword,
    resetPassword: resetPassword,
    verify: verify,
    logout: logout,
    generateAI: generateAI,
    getUser: getUser,
    getInstallId: getInstallId,
    getFingerprint: getFingerprint,
    hasCredits: hasCredits,
    isPremium: isPremium
  };
})();
